<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\PartnerInvoice;
use App\Models\StudentRegistration;
use Illuminate\Http\Request;
use Dompdf\Dompdf;
use Dompdf\Options;

class PartnerInvoiceController extends Controller
{
    public function index()
    {
        $auth_id = auth('web')->id();

        $invoices = PartnerInvoice::where('partner_id', $auth_id)->get()->map(function ($invoice) {
            $students = StudentRegistration::where('invoice_id', $invoice->id)->get();
            $invoice->students_count = $students->count();

            // Determine the overall payment status based on students' status
            // $hasUnpaid = $students->contains(function ($student) {
            //     return $student->payment_status === 'paynow';
            // });
            // $invoice->payment_status = $hasUnpaid ? 'paynow' : 'paid';

            return $invoice;
        });

        $data = [
            'title' => 'Student Invoice',
            'sidebar_menu' => 'partner_invoice',
            'invoices' => $invoices
        ];

        return view('client.partner_invoice.index')->with($data);
    }

    function paid_invoice($id){
        $invoice = PartnerInvoice::hashidfind($id);
        $invoice->payment_status = 'paid';
        $invoice->save();
        return \Redirect::back();
    }
    public function view_invoice($id)
    {
        // Encode the ID using hashids_encode()
        $encodedId = hashids_decode($id);
        $students = StudentRegistration::with('Course')->where('invoice_id', $encodedId)->get();
        $data = [
            'title' => 'View Student Invoice',
            'sidebar_menu' => 'view_invoice',
            'students' => $students,
        ];
        return view('client.partner_invoice.view_invoice')->with($data);
    }
    public function save_modal_data(Request $request)
    {
        $student = StudentRegistration::find($request->student_id);
        if ($student) {
            $student->marks = $request->marks;

            // Update status based on marks
            if ($request->marks < 45) {
                $student->status = 'fail';
                $student->certification_number = null;
            } else {
                $student->status = 'pass';

                // Generate a unique certification number
                $student->certification_number = $this->generateNumber();
            }
            $student->save();
            $msg = [
                'success' => 'Student data updated successfully.',
                'reload' => true,
            ];
            return response()->json($msg);

        }
        return ['error' => 'Student not found.'];
    }
    private function generateNumber()
    {
        $prefix = 'cert';
        do {
            $certificationNumber = str_pad(mt_rand(1, 999999), 6, '0', STR_PAD_LEFT);
            $certificationNumber = $certificationNumber;
            $exists = StudentRegistration::where('certification_number', $certificationNumber)->exists();
        } while ($exists);

        return $certificationNumber;
    }

    public function download_certificate($id)
    {
        $student = StudentRegistration::findOrFail($id);

        // Prepare data for the certificate view
        $data = [
            'student' => $student,
            'title' => 'Certificate of Completion',
        ];
        // Render the HTML content with the student data
        // return view('client.partner_invoice.certificate', $data);
        $pdfContent = view('client.partner_invoice.certificate', $data)->render();

        // Initialize Dompdf with options
        $options = new Options();
        $options->set('isHtml5ParserEnabled', true);
        $options->set('isPhpEnabled', true);

        $dompdf = new Dompdf($options);

        $dompdf->setPaper('A4', 'portrait');

        $dompdf->loadHtml($pdfContent);

        $dompdf->render();

        $pdfPath = 'pdfs/' . 'certificate-' . $student->id . '.pdf';
        file_put_contents(storage_path('app/' . $pdfPath), $dompdf->output());

        return response()->download(storage_path('app/' . $pdfPath));
    }
    // verification certificate
    public function verify_certification()
    {
        return view('client.partner_invoice.verify_certification');
    }
    public function handle_verification(Request $request)
    {
        $request->validate([
            'certification_number' => 'required|string|exists:student_registrations,certification_number',
        ]);

        $student = StudentRegistration::where('certification_number', $request->certification_number)->firstOrFail();

        // Redirect to the certificate view
        return redirect()->route('certificate', ['id' => $student->id]);
    }
    public function show_certificate($id)
    {
        $student = StudentRegistration::findOrFail($id);
        return view('client.partner_invoice.certificate', compact('student'));
    }
}
