<?php

namespace App\Services;

use App\Models\Admin;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;

class GoogleAdsService
{
    protected $baseUrl = 'https://googleads.googleapis.com/v16';
    protected $client;

    public function __construct()
    {
        $this->initializeClient();
    }

    private function initializeClient()
    {
        $admin = Admin::find(1);
        $authorization = json_decode($admin->google_access_token, true);
        $accessToken = $authorization['access_token'];

        $this->client = new Client([
            'headers' => [
                'Authorization' => "Bearer $accessToken",
                'Content-Type' => 'application/json',
                'developer-token' => config('services.google.developer_token'),
            ]
        ]);
    }

    private function refreshToken()
    {
        $admin = Admin::find(1);
        $refreshToken = $admin->google_refresh_token;
        $clientId = config('services.google.client_id');
        $clientSecret = config('services.google.client_secret');

        $client = new Client();
        $response = $client->post('https://oauth2.googleapis.com/token', [
            'form_params' => [
                'client_id' => $clientId,
                'client_secret' => $clientSecret,
                'grant_type' => 'refresh_token',
                'refresh_token' => $refreshToken,
            ],
        ]);

        $newTokenData = json_decode($response->getBody(), true);
        $admin->google_access_token = json_encode($newTokenData);
        $admin->save();

        $this->initializeClient();
    }

    private function handleRequest($method, $url, $options = [])
    {
        try {
            $response = $this->client->{$method}($url, $options);
        } catch (RequestException $e) {
            if ($e->getResponse()->getStatusCode() == 401) {
                $this->refreshToken();
                $response = $this->client->{$method}($url, $options);
            } else {
                throw $e;
            }
        }
        return $response;
    }

    public function getCustomers()
    {
        $response = $this->handleRequest('get', "$this->baseUrl/customers:listAccessibleCustomers");
        return json_decode($response->getBody(), true);
    }

    function createCampaignBudget($customerId, $requestBody = null)
    {
        $response = $this->handleRequest('post', "$this->baseUrl/customers/{$customerId}/campaignBudgets:mutate", [
            'json' => [
                'operations' => [
                    [
                        'create' => [
                            'name' =>  $requestBody['campaign_name'],
                            'amountMicros' => 50000000,
                            'deliveryMethod' => 'STANDARD',
                        ],
                    ],
                ],
            ],
        ]);

        $data = json_decode($response->getBody(), true);
        return $data['results'][0]['resourceName'];
    }

    public function getCampaigns($customerId)
    {
        $response = $this->handleRequest('post', "$this->baseUrl/customers/$customerId/googleAds:search", [
            'json' => [
                'query' => "SELECT campaign.id, campaign.name, campaign.start_date, campaign.end_date, campaign.target_roas.target_roas, campaign_budget.amount_micros, campaign.status, metrics.clicks, metrics.impressions, metrics.ctr, metrics.conversions, metrics.cost_micros FROM campaign WHERE campaign.status != 'REMOVED'"
            ]
        ]);

        return json_decode($response->getBody(), true);
    }

    public function getCampaignByResourceName($customerId, $campaignResourceName)
    {
        $response = $this->handleRequest('post', "$this->baseUrl/customers/$customerId/googleAds:search", [
            'json' => [
                'query' => "SELECT campaign.id, campaign.name, campaign.status, campaign.start_date, campaign.end_date, campaign.advertising_channel_type, metrics.cost_micros FROM campaign WHERE campaign.resource_name = '$campaignResourceName'"
            ],
        ]);

        $data = json_decode($response->getBody(), true);
        return $data['results'][0];
    }

    function createCampaign($customerId, $campaignBudgetResourceName, $requestBody = null)
    {
        if ($requestBody['campaign_type'] === 'SEARCH') {
            $operations = [
                [
                    'create' => [
                        'name' => $requestBody['campaign_name'],
                        'campaignBudget' => $campaignBudgetResourceName,
                        'advertisingChannelType' => $requestBody['campaign_type'],
                        'status' => 'ENABLED',
                        // 'advertisingChannelSubType' => 'LEAD_FORM',
                        'biddingStrategyType' => 'MAXIMIZE_CONVERSIONS',
                        'maximizeConversions' => [
                            'targetCpaMicros' => 5000000,
                        ],
                        'startDate' => $requestBody['start_date'],
                        'endDate' => $requestBody['end_date'],
                        // 'networkSettings' => [
                        //     'targetGoogleSearch' => true,
                        //     'targetSearchNetwork' => true,
                        //     'targetContentNetwork' => false,
                        //     'targetPartnerSearchNetwork' => false,
                        // ],
                        'targetingSetting' => [
                            'targetRestrictions' => [
                                [
                                    'targetingDimension' => 'AUDIENCE',
                                    'bidOnly' => false,
                                ]
                            ],
                        ],
                        // 'languages' => [
                        //     [
                        //         'languageConstant' => 'languageConstants/1000'  // English
                        //     ]
                        // ]
                    ],
                ],
            ];
        } elseif ($requestBody['campaign_type'] === 'PERFORMANCE_MAX') {
            $operations = [];
        }

        $response = $this->handleRequest('post', "$this->baseUrl/customers/{$customerId}/campaigns:mutate", [
            'json' => [
                'operations' => $operations,
            ],
        ]);

        $data = json_decode($response->getBody(), true);
        return $data['results'][0]['resourceName'];
    }

    public function getAdGroups($customerId)
    {
        $response = $this->handleRequest('post', "$this->baseUrl/customers/$customerId/googleAds:search", [
            'json' => [
                'query' => "SELECT ad_group.id, ad_group.name, ad_group.status, metrics.impressions, metrics.clicks, metrics.ctr, metrics.cost_micros, metrics.conversions FROM ad_group WHERE ad_group.status != 'PAUSED'"
            ],
        ]);

        return json_decode($response->getBody(), true);
    }

    function createAdGroup($customerId, $campaignResourceName, $requestBody = null)
    {
        $response = $this->handleRequest('post', "$this->baseUrl/customers/{$customerId}/adGroups:mutate", [
            'json' => [
                'operations' => [
                    [
                        'create' => [
                            'name' => $requestBody['campaign_name'],
                            'campaign' => $campaignResourceName,
                            'status' => 'ENABLED',
                            'cpcBidMicros' => $requestBody['cpc_bid'],
                        ],
                    ],
                ],
            ],
        ]);

        $data = json_decode($response->getBody(), true);
        return $data['results'][0]['resourceName'];
    }

    function addKeywordToAdGroup($customerId, $adGroupResourceName, $requestBody = null)
    {
        $operations = [];
        foreach ($requestBody['keywords'] as $keyword) {
            array_push($operations, [
                [
                    'create' => [
                        'adGroup' => $adGroupResourceName,
                        'keyword' => [
                            'text' => $keyword,
                            'matchType' => $requestBody['keyword_match_types']
                        ],
                        'status' => 'ENABLED',
                    ]
                ]
            ]);
        }

        $response = $this->handleRequest('post', "$this->baseUrl/customers/{$customerId}/adGroupCriteria:mutate", [
            'json' => [
                'operations' => $operations,
            ]
        ]);

        return json_decode($response->getBody(), true);
    }

    public function getAds($customerId)
    {
        $response = $this->handleRequest('post', "$this->baseUrl/customers/$customerId/googleAds:search", [
            'json' => [
                'query' => "SELECT campaign.id, campaign.name, campaign.advertising_channel_type, ad_group.id, ad_group.name, ad_group_ad.ad.id, ad_group_ad.ad.name, ad_group_ad.ad.final_urls, ad_group_ad.status, metrics.impressions, metrics.clicks, metrics.ctr, metrics.cost_micros, metrics.average_cpc, metrics.average_cpm, metrics.conversions FROM ad_group_ad"
            ],
        ]);

        return json_decode($response->getBody(), true);
    }

    function createAd($customerId, $adGroupResourceName, $requestBody = null)
    {
        $headlines = [];
        foreach ($requestBody['ad_headlines'] as $headline) {
            if ($headline) {
                array_push($headlines, [
                    'text' => $headline,
                ]);
            }
        }

        $descriptions = [];
        foreach ($requestBody['ad_descriptions'] as $description) {
            if ($description) {
                array_push($descriptions, [
                    'text' => $description,
                ]);
            }
        }

        $response = $this->handleRequest('post', "https://googleads.googleapis.com/v16/customers/{$customerId}/adGroupAds:mutate", [
            'json' => [
                'operations' => [
                    [
                        'create' => [
                            'adGroup' => $adGroupResourceName,
                            'status' => 'ENABLED',
                            'ad' => [
                                'finalUrls' => [
                                    $requestBody['ad_url']
                                ],
                                'responsiveSearchAd' => [
                                    'headlines' => $headlines,
                                    'descriptions' => $descriptions,
                                ],
                            ],
                        ],
                    ],
                ],
            ],
        ]);

        $data = json_decode($response->getBody(), true);
        return $data['results'][0]['resourceName'];
    }

    function getConversionActions($customerId)
    {
        $response = $this->handleRequest('post', "https://googleads.googleapis.com/v16/customers/{$customerId}/googleAds:search", [
            'json' => [
                'query' => "SELECT conversion_action.resource_name, conversion_action.id, conversion_action.name, conversion_action.status, conversion_action.type FROM conversion_action"
            ],
        ]);

        return json_decode($response->getBody(), true);
    }

    function createConversionAction($customerId, $requestBody = null)
    {
        $response = $this->handleRequest('post', "https://googleads.googleapis.com/v16/customers/{$customerId}/conversionActions:mutate", [
            'json' => [
                'operations' => [
                    [
                        'create' => [
                            'name' => $requestBody['name'],
                            'type' => $requestBody['type'],
                            'category' => $requestBody['category'],
                            'status' => 'ENABLED',
                            'valueSettings' => [
                                'defaultValue' => number_format((float) $requestBody['value'], 1, '.', ''),
                                'alwaysUseDefaultValue' => true,
                            ],
                            'countingType' => $requestBody['counting_type'],
                            'clickThroughLookbackWindowDays' => $requestBody['click_through_days'],
                            'viewThroughLookbackWindowDays' => $requestBody['view_through_days']
                        ]
                    ]
                ]
            ]
        ]);

        return json_decode($response->getBody(), true);
    }
}
