<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\PartnerInvoice;
use App\Models\User;
use Illuminate\Http\Request;
use App\Models\UserDeviceToken;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use App\Models\StudentRegistration;

class DashboardController extends Controller
{


    public function __construct()
    {
        $this->middleware('auth:web');
    }

    public function index()
    {
        $auth_id = auth('web')->id();

        // Option 1: Using Raw Query
        // $StudentCount = DB::table('partner_invoices')
        //     ->join('student_registrations', 'partner_invoices.id', '=', 'student_registrations.invoice_id')
        //     ->where('partner_invoices.partner_id', $auth_id)
        //     ->count('student_registrations.id');

        // Option 2: Using Eloquent Relationships
        $StudentCount = PartnerInvoice::where('partner_id', $auth_id)
            ->withCount('studentRegistrations')
            ->get()
            ->sum('student_registrations_count');

        $data = array(
            'breadcrumb' => 'Dashboard',
            'title' => 'Dashboard',
            'main_wallet_bls' => '',
            'sub_accounts' => '',
            'remaining_balance' => $remaining_amount ?? '',
            'total_ppc_leads' => $this_week_client_leads ?? '',
            'this_week_payment' => $this_week_payment ?? '',
            'dates_text' => "From:",
            'sidebar_menu' => 'dashboard',
            'StudentCount' => $StudentCount,
        );

        return view('client.dashboard')->with($data);
    }

    public function edit_profile()
    {
        $data = array(
            'breadcrumb' => 'Profile',
            'title' => 'Profile',
            'edit' => auth('web')->user(),
            'sub_title' => 'Update Password',
            'sidebar_menu' => 'password',
        );
        return view('client.profile')->with($data);
    }

    public function update_profile(Request $request)
    {

        $rules = [
            'full_name' => 'required|string|max:50',
            'date_of_birth' => 'required',
            'profession' => 'required',
            'experience' => 'required',
            'country' => 'required',
            'email' => 'required|unique:users,email,'.$request->id,
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return ['errors' => $validator->errors()];
        }

        try {

            DB::beginTransaction();

            $user = User::hashidFind($request->id);

            $user->full_name = $request->full_name;
            $user->date_of_birth = $request->date_of_birth;
            $user->profession = $request->profession;
            $user->experience = $request->experience;
            $user->country = $request->country;
            $user->email = $request->email;
            $user->user_name = 'user-'.strtolower($request->full_name);

            $user->save();

            $msg = [
                'success' => 'Profile Updated Successfully',
                'reload' => true,
            ];
            DB::commit();
            return response()->json($msg);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Update Profile: ' . $e->getMessage()], 500);
        }
    }


    public function update_password(Request $request)
    {
        // dd($request);

        $rules = [
            'current_password' => 'required',
           'password' => 'required|min:8|max:12|confirmed',
        ];

        $messages = [
            'password.regex' => 'Invalid Format. Password should be 8 characters, with at least 1 number and special characters.',
        ];

        $validator = Validator::make($request->all(), $rules, $messages);

        if ($validator->fails()) {
            return ['errors' => $validator->errors()];
        }

        $user = User::hashidFind($request->id);
        // dd($user);
        if (!Hash::check($request->current_password, $user->password)) {
           return [ 'error' => 'Current Password Not Match', ];
        } else {
            $user->password = Hash::make($request->password);
            $user->save();
            $msg = [
                'success' => 'User password has been updated',
                'reload' => true,
            ];
        }
        return response()->json($msg);
    }



    public function edit_user_profile()
    {
        $data = array(
            'breadcrumb' => 'User Profile',
            'title' => 'Profile',
            'edit' => auth('web')->user(),
            'sidebar_menu' => "profile",
            'sub_title' => 'Edit Profile',
        );

        return view('client.individual_user_profile.profile')->with($data);
    }

    public function update_user_profile(Request $request)
    {

        $rules = [
            'full_name' => 'required|string|max:50',
            'date_of_birth' => 'required',
            'profession' => 'required',
            'experience' => 'required',
            'country' => 'required',
            // 'email' => 'required|unique:users,email,'.$request->id,
        ];
        if (isset($request->profile_image) && !empty($request->profile_image)) {
            $rules['profile_image'] = 'image|mimes:png,jpeg,jpg|max:2048';
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return ['errors' => $validator->errors()];
        }

        try {

            DB::beginTransaction();

            $user = User::hashidFind($request->id);

            $user->full_name = $request->full_name;
            $user->date_of_birth = $request->date_of_birth;
            $user->profession = $request->profession;
            $user->experience = $request->experience;
            $user->country = $request->country;
            $user->email = $request->email;
            $user->user_name = 'user-'.strtolower($request->full_name);

            if ($request->hasFile('profile_image')) {
                $profile_img = uploadSingleFile($request->file('profile_image'), 'uploads/client/profile_images/', 'png,jpeg,jpg');
                if (is_array($profile_img)) {
                    return response()->json($profile_img);
                }
                if (file_exists($user->profile_image)) {
                    @unlink($user->profile_image);
                }
                $user->profile_image = $profile_img;
            }

            $user->save();

            $msg = [
                'success' => 'Profile Updated Successfully',
                'reload' => true,
            ];
            DB::commit();
            return response()->json($msg);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Update Profile: ' . $e->getMessage()], 500);
        }
    }



}
